Unit Controls;
{-$define event_log}
Interface

uses
{$ifdef linux}
 Xlib,
{$endif}
 Windows,Messages,Classes,SysUtils,Graphics;

Type
 TControl=class(TComponent)
 private
  EOnClick:TNotifyEvent;
 protected
  procedure ReadProperty(Name:string; Reader:TReader); override;
  Function FindMethod(Name:string):TMethod;
 public
 end;

 TWinControl=class;

 TCustomControl=class(TControl)
 private
  fParent:TWinControl;
  fCanvas:TCanvas;
  fFont:TFont;
  fColor:integer;
  fParentFont:boolean;
  EOnPaint:TNotifyEvent;
 protected
  fLeft,fTop:integer;
  fWidth,fHeight:integer;
  fCaption:string;
  fTransparent:boolean;
  procedure ReadProperty(Name:string; Reader:TReader); override;
  Function SubProperty(Name:string):TPersistent; override;
  Function GetFont:TFont;
  Procedure SetFont(Value:TFont);
  Function GetCanvas:TCanvas;
 public
  Constructor Create(AOwner:TComponent); override;
  Destructor Destroy; override;
  procedure Loaded; override;
  procedure Paint; virtual;
  function ClientRect:TRect;
  property Font:TFont read GetFont write SetFont;
  property Canvas:TCanvas read GetCanvas;
  property Parent:TWinControl read fParent;
  property Color:integer read fColor write fColor;
 end;

 TGraphicControl=class(TCustomControl)
 protected
  procedure SetParentComponent(Value:TComponent); override;
 public
  procedure Loaded; override;
 end;

 TControlStyle = set of (csAcceptsControl, csCaptureMouse, csClickEvents, csFramed,
  csSetCaption, csOpaque, cdDoubleClicks);
 TControlState = set of (csLButtonDown, csClicked, csPalette, csReadingState, csAlignmentNeeded, csFocusing, csCreating, csPaintCopy, csCustomPaint, csDestroyingHandle, csDocking);
 TBorderStyle=(bsNone,bsSingle);

 TWinControl=class(TCustomControl)
 private
  fGraphics:TList;
 {$ifdef win32}
  procedure WMSize(Var Msg:TWMSize); message wm_size;
  procedure WMLButtonDown(Var msg:TWMLButtonDown); message wm_lbuttondown;
  procedure WMLButtonUp(Var msg:TWMLButtonUp); message wm_lbuttonup;
  procedure WMEraseBkGnd(Var msg:TWMEraseBkGnd); message wm_erasebkgnd;
  procedure WMDestroy(Var Msg:TWMDestroy); message wm_destroy;
  procedure WMPaint(Var Msg:TWMPaint); message wm_paint;
 public
  procedure DefaultHandler(Var Message); override;
 {$endif}
 {$ifdef linux}
  procedure ButtonPress(Var Event:XButtonEvent); message ButtonPress;
  procedure ButtonRelease(Var Event:XButtonEvent); message ButtonRelease;
  procedure MotionNotify(Var Event:XMotionEvent); message MotionNotify;
  procedure Expose(Var Event:XExposeEvent); message expose;
 {$ifdef event_log}
  procedure DefaultHandler(Var Message); override;
 {$endif}
 {$endif}
 protected
  fHandle:integer;
  fOldProc:integer;
  fTabOrder:integer;
  fControlStyle:TControlStyle;
  fControlState:TControlState;
  fControls:TList;
  procedure ReadProperty(Name:string; Reader:TReader); override;
  procedure SetParentComponent(Value:TComponent); override;
  function GetParentComponent:TComponent; override;
  procedure HandleNeeded;
  procedure CreateHandle; virtual; abstract;
  Procedure SetHandle(Value:integer);
  procedure SetText(Value:string); virtual;
  procedure AddChild(AChild:TWinControl);
 public
  Constructor Create(AOwner:TComponent); override;
  Destructor Destroy; override;
  procedure Paint; override;
  procedure Show;
  property Handle:integer read fHandle write SetHandle;
 end;

Implementation

function WndProc(Hwnd,Msg,wParam,lParam:integer):integer; stdcall;
{$ifdef win32}
 var
  obj:TObject;
  dsp:TMessage;
 begin
  obj:=TObject(GetProp(HWnd,'MySoft.LightVCL'));
  if not Assigned(obj) then
   Result:=DefWindowProc(HWnd,Msg,wParam,lParam)
  else begin
   dsp.msg:=msg;
   dsp.wParam:=WParam;
   dsp.lParam:=lParam;
   dsp.result:=0;
   obj.Dispatch(dsp); //Perform(Msg,wParam,lParam);
   Result:=dsp.result;
  end;
{$else}
 begin
{$endif}
 end;

procedure TControl.ReadProperty(Name:string; Reader:TReader);
 Const
  TControlProperties:array[0..0] of PChar=(
   'OnClick'
  );
 begin
  case StringIndex(Name,TControlProperties) of
   0 : TMethod(EOnClick):=FindMethod(Reader.StringProperty);
   else inherited;
  end;
 end;

Function TControl.FindMethod(Name:string):TMethod;
 var
  Cmp:TComponent;
 begin
  Cmp:=Self;
  Result.code:=nil;
  Result.data:=Self;
  while (Result.code=nil)and(Cmp<>nil) do begin
   result.data:=cmp;
   result.code:=cmp.MethodAddress(Name);
   Cmp:=Cmp.Owner;
  end;
  if Result.code=nil then msgbox(Name+' non trouv chez '+fName);
 end;

Constructor TCustomControl.Create(AOwner:TComponent);
 begin
  inherited;
  fTransparent:=False;
  fParentFont:=True;
 end;

Destructor TCustomControl.Destroy;
 begin
  fFont.Free;
  fCanvas.Free;
  inherited;
 end;

procedure TCustomControl.Loaded;
 begin
  inherited;
//  if Assigned(fParent) then Font.Assign(fParent.Font);
 end;

function TCustomControl.GetCanvas:TCanvas;
 begin
  if fCanvas=nil then fCanvas:=TCanvas.Create;//(Self);
  Result:=fCanvas;
 end;

procedure TCustomControl.ReadProperty(Name:string; Reader:TReader);
 Const
  TWinControlProperties:array[0..8] of PChar=(
   'Left','Top',
   'Width','Height',
   'Color','Transparent',
   'Caption',
   'OnPaint',
   'ParentFont'
  );
 begin
  case StringIndex(Name,TWinControlProperties) of
    0 : fLeft:=Reader.IntegerProperty;
    1 : fTop:=Reader.IntegerProperty;
    2 : fWidth:=Reader.IntegerProperty;
    3 : fHeight:=Reader.IntegerProperty;
    4 : fColor:=Reader.ColorProperty;
    5 : fTransparent:=Reader.BooleanProperty;
    6 : fCaption:=Reader.StringProperty;
    7 : TMethod(EOnPaint):=FindMethod(Reader.StringProperty);
    8 : fParentFont:=Reader.BooleanProperty;
   else inherited;
  end;
 end;

Function TCustomControl.SubProperty(Name:string):TPersistent;
 Const
  TControlSubProperties:array[0..0] of PChar=(
   'Font'
  );
 begin
  case StringIndex(Name,TControlSubProperties) of
   0 : begin if fFont=nil then fFont:=TFont.Create; Result:=fFont; end;
   else Result:=nil;
  end;
 end;

Function TCustomControl.GetFont:TFont;
 begin
  if fFont=nil then begin
   if fParentFont and (Parent<>nil) then begin
    Result:=Parent.Font;
    exit;
   end;
   fFont:=TFont.Create;
  end;
  Result:=fFont;
 end;

Procedure TCustomControl.SetFont(Value:TFont);
 begin
  Font.Assign(Value);
 end;

Procedure TCustomControl.Paint;
 begin
  if not fTransparent then
   with Canvas do begin
   {$ifdef linux}
    Brush.Color:=Self.Color;
    FillRect(Rect(0,0,fWidth,fHeight));
   {$endif}
    if Assigned(EOnPaint) then EOnPaint(Self);
   end;
 end;

Function TCustomControl.ClientRect:TRect;
 begin
  Result.Left:=fLeft;
  Result.Top:=fTop;
  Result.Right:=fLeft+fWidth;
  Result.Bottom:=fTop+fHeight;
 end;

Procedure TGraphicControl.Loaded;
 begin
  inherited;
  Canvas.SetOrg(fLeft,fTop);
 end;

procedure TGraphicControl.SetParentComponent(Value:TComponent);
 begin
  while (Value<>nil)and not(Value is TWinControl) do Value:=Value.ParentComponent;
  if Value<>nil then begin
   fParent:=TWinControl(Value);
   Font.Assign(fParent.Font);
   fParent.fGraphics.Add(Self);
  end;
 end;

Constructor TWinControl.Create(AOwner:TComponent);
 begin
  inherited;
  fControls:=TList.Create;
  fGraphics:=TList.Create;
  fColor:=clSilver;
 end;

Destructor TWinControl.Destroy;
 begin
  if fColor<>0 then DeleteObject(fColor);
  Handle:=0;
  fControls.Free;
  fGraphics.Free;
  inherited;
 end;

procedure TWinControl.ReadProperty(Name:string; Reader:TReader);
 Const
  TWinControlProperties:array[0..1] of PChar=(
   'Text',
   'TabOrder'
  );
 begin
  case StringIndex(Name,TWinControlProperties) of
    0 : SetText(Reader.StringProperty);
    1 : fTabOrder:=Reader.IntegerProperty;
   else inherited;
  end;
 end;

procedure TWinControl.SetParentComponent(Value:TComponent);
 begin
  while (Value<>nil)and not(Value is TWinControl) do Value:=Value.ParentComponent;
  if Value<>nil then begin
   fParent:=TWinControl(Value);
   Canvas.Font.Assign(fParent.Font);
   fParent.AddChild(Self);
  end;
 end;

Function TWinControl.GetParentComponent:TComponent;
 begin
  Result:=fParent;
 end;

procedure TWinControl.HandleNeeded;
 begin
  if fParent<>nil then fParent.HandleNeeded;
  if fHandle=0 then CreateHandle;
 end;

Function InRange(i,min,max:integer):boolean;
 begin
  result:=(i>=min)and(i<max);
 end;

Procedure TWinControl.SetHandle(Value:integer);
 begin
  if fHandle<>0 then begin
   SetWindowLong(fHandle,GWL_WNDPROC,fOldProc);
   CloseHandle(fHandle);
  end;
  fHandle:=Value;
  if fHandle<>0 then begin
   fOldProc:=GetWindowLong(fHandle,GWL_WNDPROC);
   SetProp(fHandle,'MySoft.LightVCL',Cardinal(Self));
   SetWindowLong(fHandle,GWL_WNDPROC,integer(@WndProc));
   SendMessage(fHandle,WM_SETFONT,Font.Handle,0);
  end;
 end;

Procedure TWinControl.SetText(Value:string);
 begin
  fCaption:=Value;
 end;

Procedure TWinControl.AddChild(AChild:TWinControl);
 begin
  fControls.Add(AChild);
 end;

Procedure TWinControl.Show;
 var
  i:integer;
 begin
  HandleNeeded;
  for i:=0 to fControls.Count-1 do TWinControl(fControls[i]).Show;
  ShowWindow(fHandle,SW_SHOW);
 end;

{$ifdef win32} //----------------------------------------------------------WIN32
procedure TWinControl.WMSize(Var Msg:TWMSize);
 begin
  inherited;
//  if Msg.SizeType<>0 then begin
   fWidth:=msg.Width;
   fHeight:=msg.Height;
//  end;
 end;

procedure TWinControl.WMLButtonDown(Var msg:TWMLButtonDown);
 begin
  inherited;
  Include(fControlState,csClicked);
 end;

procedure TWinControl.WMLButtonUp(Var msg:TWMLButtonUp);
 begin
  inherited;
  if csClicked in fControlState then begin
   if  Assigned(EOnClick)
   and InRange(msg.XPos,0,fWidth)
   and InRange(msg.YPos,0,fHeight)
   then EOnClick(Self);
   Exclude(fControlState,csClicked);
  end;
 end;

procedure TWinControl.WMEraseBkGnd(Var msg:TWMEraseBkGnd);
 begin
  msg.Result:=1;
 {$ifdef win32}
  if not fTransparent then
   with Canvas do begin
    Handle:=Msg.DC;
    Brush.Color:=Self.Color;
    FillRect(Rect(0,0,fWidth,fHeight));
   end;
 {$endif}
 end;

procedure TWinControl.WMPaint(Var Msg:TWMPaint);
 begin
  Canvas.Handle:=Msg.DC;
  if Canvas.Handle=0 then Canvas.Handle:=GetDC(fHandle);
  Paint;
  if Msg.DC=0 then ReleaseDC(fHandle,Canvas.Handle);
  inherited;
 end;

procedure TWinControl.WMDestroy(Var Msg:TWMDestroy);
 begin
  inherited;
  PostQuitMessage(0);
 end;

procedure TWinControl.DefaultHandler(Var Message);
 begin
  with TMessage(Message) do Result:=CallWindowProc(pointer(fOldProc),fHandle,Msg,wParam,lParam)
 end;
{$endif}

{$ifdef linux}//-----------------------------------------------------------LINUX
procedure TWinControl.ButtonPress(Var Event:XButtonEvent);
 begin
  inherited;
  Include(fControlState,csLButtonDown);
  if csClickEvents in fControlStyle then begin
   Include(fControlState,csClicked);
   Paint;
  end;
 end;

procedure TWinControl.ButtonRelease(Var Event:XButtonEvent);
 begin
  inherited;
  if csClicked in fControlState then begin
   if  Assigned(EOnClick)
   and InRange(Event.x,0,fWidth)
   and InRange(Event.y,0,fHeight)
   then EOnClick(Self);
   Exclude(fControlState,csClicked);
   Paint;
  end;
  Exclude(fControlState,csLButtonDown);
 end;

procedure TWinControl.MotionNotify(Var Event:XMotionEvent);
 var
  cs:TControlState;
 begin
  if (csLButtonDown in fControlState)
  and(csClickEvents in fControlStyle) then begin
   cs:=fControlState;
   if  InRange(Event.x,0,fWidth)
   and InRange(Event.y,0,fHeight) then
    Include(cs,csClicked)
   else
    Exclude(cs,csClicked);
   if cs<>fControlState then begin
    fControlState:=cs;
    Paint;
   end;
  end;
 end;

procedure TWinControl.Expose(Var Event:XExposeEvent);
 begin
  if Event.count<>0 then exit;
  fWidth :=Event.x+Event.width;
  fHeight:=Event.y+Event.height;
  Canvas.Handle:=fHandle;
  Paint;
 end;
{$ifdef event_log}
procedure TWinControl.DefaultHandler(Var Message);
 begin
  with XEvent(Message) do WriteLn(xtype);
 end;
{$endif}
{$endif}

procedure TWinControl.Paint;
 var
  i:integer;
 begin
  inherited;
  for i:=0 to fGraphics.Count-1 do
   with TGraphicControl(fGraphics[i]) do begin
    Canvas.Handle:=Self.Canvas.Handle;
    Paint;
   end;
 end;

end.
