Unit Graphics;

Interface

uses
{$ifdef linux}
 Xlib,
{$endif}
 Windows,classes,sysutils,dialogs;

Type
 TFontStyle = (fsBold, fsItalic, fsUnderline, fsStrikeOut);
 TFontStyles = set of TFontStyle;

 TFont=class(TPersistent)
 private
  fHandle :integer;
  fCharSet:string;
  fColor  :integer;
  fHeight :integer;
  fName   :string;
  fStyle  :TFontStyles;
  function GetHandle:integer;
 public
  Destructor Destroy; override;
  Procedure ReadProperty(Name:string; Reader:TReader); override;
  Procedure Assign(AFont:TFont);
  Property Handle:integer read GetHandle;
  Property Color:integer read fColor write fColor;
 end;

 TPen=class
 private
  fColor:integer;
 public
  property Color:integer read fColor write fColor;
 end;

 TBrushStyle=(bsSolid, bsClear, bsHorizontal, bsVertical, bsFDiagonal, bsBDiagonal, bsCross, bsDiagCross);

 TBrush=class
 private
  fColor:integer;
  fStyle:TBrushStyle;
 {$ifdef win32}
  fHandle:integer;
  function GetHandle:integer;
  procedure SetHandle(value:integer);
  procedure SetColor(Value:integer);
  property Handle:integer read GetHandle write SetHandle;
 public
  Destructor Destroy; override;
 {$endif}
 public
  property Color:integer read fColor write {$ifdef win32}SetColor{$else}fColor{$endif};
  property Style:TBrushStyle read fStyle write fStyle;
 end;

 TCanvas=class//(TComponent)
 private
  fHandle:integer;
  fPen:TPen;
  fBrush:TBrush;
  fFont :TFont;
  lx,ly:integer;
  fOrgX:integer;
  fOrgY:integer;
  function GetBrush:TBrush;
  function GetPen:TPen;
  function GetFont:TFont;
  procedure SetFont(Value:TFont);
 public
  Destructor Destroy; override;
  Procedure SetOrg(ox,oy:integer);
  Procedure FillRect(R:TRect);
  Procedure Rectangle(x1,y1,x2,y2:integer);
  Procedure FrameRect(var R:TRect; cl1,cl2:integer);
  Procedure MoveTo(x,y:integer);
  Procedure LineTo(x,y:integer);
  Procedure TextOut(x,y:integer;const s:string);
  Function TextWidth(s:string):integer;
  Function TextHeight(s:string):integer;
  Property Handle:integer read fHandle write fHandle;
  Property Brush:TBrush read GetBrush;
  Property Pen:TPen read GetPen;
  Property Font:TFont read GetFont write SetFont;
 end;

 TPicture=class(TPersistent)
 private
  fSize:integer;
  fData:pointer;
 public
  Destructor Destroy; override;
  Procedure ReadProperty(Name:string; Reader:TReader); override;
  Procedure DrawRect(R:TRect; Canvas:TCanvas);
 end;

Implementation

Uses
 Controls;

Const
 FontStyles:array[TFontStyle] of PChar=('fsBold','fsItalic','fsUnderline','fsStrikeOut');

Destructor TFont.Destroy;
 begin
  DeleteObject(fHandle);
  inherited;
 end;

Procedure TFont.ReadProperty(Name:string; Reader:TReader);
 const
  TFontProperties:array[0..4] of PChar=(
   'Charset','Color','Height','Name','Style'
  );
 begin
  case StringIndex(Name,TFontProperties) of
    0 : fCharset:=Reader.StringProperty;
    1 : fColor:=Reader.ColorProperty;
    2 : fHeight:=Reader.IntegerProperty;
    3 : fName:=Reader.StringProperty;
    4 : Reader.SetProperty(fStyle,FontStyles);
   else inherited;
  end;
 end;

Procedure TFont.Assign(AFont:TFont);
 begin
  fCharSet:=AFont.fCharSet;
  fColor  :=AFont.fColor;
  fHeight :=AFont.fHeight;
  fName   :=AFont.fName;
  fStyle  :=AFont.fStyle;
  DeleteObject(fHandle);
  fHandle :=0;
 end;

Function TFont.GetHandle:integer;
 var
  LogFont: TLogFont;
 begin
  if fHandle=0 then begin
   FillChar(LogFont,SizeOf(LogFont),0);
   with LogFont do begin
    lfHeight := fHeight;
    lfWidth := 0; { have font mapper choose }
    lfEscapement := 0; { only straight fonts }
    lfOrientation := 0; { no rotation }
    if fsBold in fStyle then
      lfWeight := FW_BOLD
    else
      lfWeight := FW_NORMAL;
    lfItalic := Byte(fsItalic in fStyle);
    lfUnderline := Byte(fsUnderline in fStyle);
    lfStrikeOut := Byte(fsStrikeOut in fStyle);
    lfCharSet := DEFAULT_CHARSET;//Byte(Font.Charset);
    Move(fName[1],lfFaceName,Length(fName));
    lfQuality := DEFAULT_QUALITY;
    { Everything else as default }
    lfOutPrecision := OUT_DEFAULT_PRECIS;
    lfClipPrecision := CLIP_DEFAULT_PRECIS;
    lfPitchAndFamily := DEFAULT_PITCH;
   end;
   fHandle:=CreateFontIndirect(LogFont);
  end;
  Result:=fHandle;
 end;

{$ifdef win32}
Destructor TBrush.Destroy;
 begin
  Handle:=0;
  inherited;
 end;

function TBrush.GetHandle:integer;
 begin
  if fHandle=0 then begin
   case fStyle of
    bsClear : ;
    bsSolid : fHandle:=CreateSolidBrush(fColor);
   end;
  end;
  Result:=fHandle;
 end;


procedure TBrush.SetHandle(value:integer);
 begin
  if Value=fHandle then exit;
  if fHandle<>0 then DeleteObject(fHandle);
  fHandle:=Value;
 end;

procedure TBrush.SetColor(value:integer);
 begin
  if Value=fColor then exit;
  Handle:=0;
  fColor:=Value;
 end;
{$endif}

Destructor TCanvas.Destroy;
 begin
  fPen.Free;
  fBrush.Free;
  fFont.Free;
  inherited;
 end;

Procedure TCanvas.SetOrg(ox,oy:integer);
 begin
  fOrgX:=ox;
  fOrgY:=oy;
 end;

Function TCanvas.GetBrush:TBrush;
 begin
  if fBrush=nil then fBrush:=TBrush.Create;
  Result:=fBrush;
 end;

Function TCanvas.GetPen:TPen;
 begin
  if fPen=nil then fPen:=TPen.Create;
  Result:=fPen;
 end;

Function TCanvas.GetFont:TFont;
 begin
  if fFont=nil then fFont:=TFont.Create;
  Result:=fFont;
 end;

Procedure TCanvas.SetFont(Value:TFont);
 begin
  Font.Assign(Value);
 end;

Procedure TCanvas.FillRect(R:TRect);
 begin
 {$ifdef linux}
  XSetForeground(gDisplay,gGC,GetSysColor(Brush.fColor));
  XFillRectangle(gDisplay,fHandle,gGC,fOrgX+R.Left,fOrgY+R.Top,fOrgX+R.Right-R.Left+1,fOrgY+R.Bottom-R.Top+1);
 {$endif}
 {$ifdef win32}
  inc(r.Left,fOrgX);
  inc(r.Top,fOrgY);
  inc(r.Right,fOrgX);
  inc(r.Bottom,fOrgY);
  Windows.FillRect(fHandle,R,Brush.Handle);
 {$endif}
 end;

Procedure TCanvas.Rectangle(x1,y1,x2,y2:integer);
 begin
 {$ifdef linux}
  XSetForeground(gDisplay,gGC,GetSysColor(Pen.fColor));
  XDrawRectangle(gDisplay,fHandle,gGC,fOrgX+x1,fOrgY+y1,fOrgX+x2-x1,fOrgY+y2-y1);
 {$endif}
 {$ifdef win32}
  Windows.Rectangle(fHandle,x1,y1,x2,y2);
 {$endif}
 end;

Procedure TCanvas.FrameRect(var R:TRect; cl1,cl2:integer);
 begin
  Pen.Color:=cl1;
  MoveTo(R.left,R.Bottom);
  LineTo(R.Left,R.Top);
  LineTo(R.Right,R.Top);
  Pen.Color:=cl2;
  LineTo(R.Right,R.Bottom);
  LineTo(R.Left,R.Bottom);
  inc(R.Left);
  inc(R.Top);
  dec(R.Right);
  dec(R.Bottom);
 end;

Procedure TCanvas.MoveTo(x,y:integer);
 begin
  lx:=fOrgX+x;
  ly:=fOrgY+y;
 end;

Procedure TCanvas.LineTo(x,y:integer);
 begin
 {$ifdef linux}
  XSetForeground(gDisplay,gGC,GetSysColor(Pen.fColor));
  XDrawLine(gDisplay,fHandle,gGC,lx,ly,fOrgX+x,fOrgY+y);
  lx:=fOrgX+x;
  ly:=fOrgY+y;
 {$endif}
 {$ifdef win32}
  Windows.LineTo(fHandle,x,y);
 {$endif}
 end;

Procedure TCanvas.TextOut(x,y:integer;const s:string);
 begin
  if s='' then exit;
 {$ifdef linux}
  inc(y,gFont.max_bounds.ascent);
  XSetForeground(gDisplay,gGC,GetSysColor(Font.fColor));
  if Brush.Style=bsClear then
   XDrawString(gDisplay,fHandle,gGC,fOrgX+x,fOrgY+y,@pchar(s)[0],length(s))
  else begin
   XSetBackground(gDisplay,gGC,GetSysColor(Brush.fColor));
   XDrawImageString(gDisplay,fHandle,gGC,fOrgX+x,fOrgY+y,@pchar(s)[0],length(s));
  end;
 {$endif}
 {$ifdef win32}
  SelectObject(fHandle,Font.Handle);
  if Brush.Style=bsClear then
   SetBkMode(fHandle,TRANSPARENT)
  else
   SetBkColor(fHandle,Brush.fColor);
  SetTextColor(fHandle,Font.fColor);
  Windows.TextOut(fHandle,fOrgX+x,fOrgY+y,PChar(s),length(s));
 {$endif}
 end;

Function TCanvas.TextWidth(s:string):integer;
 {$ifdef linux}
 begin
  Result:=XTextWidth(gFont,PChar(s),length(s));
 {$endif}
 {$ifdef win32}
 var
  Size:TSize;
 begin
  Windows.GetTextExtentPoint32(FHandle, PChar(s), Length(s), Size);
  Result:=Size.cX;
 {$endif}
 end;

Function TCanvas.TextHeight(s:string):integer;
 {$ifdef linux}
 begin
  Result:=gFont.max_bounds.ascent + gFont.max_bounds.descent;
 {$endif}
 {$ifdef win32}
 var
  Size:TSize;
 begin
  Windows.GetTextExtentPoint32(FHandle, PChar(s), Length(s), Size);
  Result:=Size.cY;
 {$endif}
 end;


Destructor TPicture.Destroy;
 begin
  FreeMem(fData);
  inherited;
 end;

Procedure TPicture.ReadProperty(Name:string; Reader:TReader);
 const
  TFontProperties:array[0..0] of PChar=(
   'Data'
  );
 begin
  case StringIndex(Name,TFontProperties) of
    0 : fData:=Reader.BinaryProperty(fSize);
   else inherited;
  end;
 end;

Procedure TPicture.DrawRect(R:TRect; Canvas:TCanvas);
{$ifdef win32} // we need a TBitmap !
 type
  TBMP=record
   ClassName:string[7]; // "TBitmap"
   Size:integer;
   FileHeader:TBitmapFileHeader;
   InfoHeader:TBitmapInfo;
  end;
 var
  BMP:^TBMP;
  Bits:pointer;
 begin
  if fSize<SizeOf(TBitmapFileHeader) then exit;
  BMP:=fData;
  if BMP.ClassName<>'TBitmap' then exit;
  if BMP.FileHeader.bfType<>$4D42 then exit;
  Bits:=pointer(integer(@BMP.FileHeader)+BMP.FileHeader.bfOffBits);
  SetDIBitsToDevice(
   Canvas.Handle, // handle of device context
   R.Left,        // x-coordinate of upper-left corner of dest. rect.
   R.Top,	  // y-coordinate of upper-left corner of dest. rect.
   BMP.InfoHeader.bmiHeader.biWidth,	// source rectangle width
   BMP.InfoHeader.bmiHeader.biHeight,	// source rectangle height
   0,	// x-coordinate of lower-left corner of source rect.
   0,	// y-coordinate of lower-left corner of source rect.
   0,	// first scan line in array
   BMP.InfoHeader.bmiHeader.biHeight,	// number of scan lines
   Bits,	// address of array with DIB bits
   BMP.InfoHeader,	// address of structure with bitmap info.
   DIB_RGB_COLORS 	// RGB or palette indices
  );
 end;
{$else}
 begin
 end;
{$endif}

end.